let config = {				                                            // Objet contenant les paramètres de notre jeu
    canvasSize : 320,     			                                    // Dimension du canvas (largeur = hauteur)
    nbCards : 32,    			                                        // Nombre de cartes sur le plateau
    scoreInitial : 50,			                                        // Score du joueur à chaque début de partie
    cardSize : {width:32, height:48},			                        // Dimension d’une carte (largeur, hauteur)     
    imagesList : ["memory.png", "cards.png", "text.png", "popup.png"]	// Liste des fichiers images nécessaires pour le jeu
};

let app = { 			                                                // Objet principal de l'application 
    canvas : document.querySelector("#app"),    	                    // Référence au canvas
    score : 0,			                                                // Score actuel du joueur
    images : {}			                                                // Images téléchargées et utilisables
};			         

app.canvas.width = config.canvasSize;                                   // On défini la largeur du canvas
app.canvas.height = config.canvasSize;                                  // On défini la hauteur du canvas
app.context = app.canvas.getContext("2d");                              // On ajoute le context 2D à l'objet app
app.context.imageSmoothingEnabled = false;                              // On active une option pour que les images ne soient pas lissées (conseillé pour le pixelart)
app.canvas.onclick = clicCanvas;                                        // On défini un événement qui se déclenche lorsque l'utilisateur cliquera sur le canvas
window.onresize = resizeCanvas;                                         // On défini un événement qui se déclenche si le navigateur change de taille
window.onload = loadImages;                                             // On charge les images dès que la page est chargée
resizeCanvas();                                                         // On appel le redimensionnement une fois pour s'assurer que le canvas est bienen plein écran

function resizeCanvas() {
    const body = document.querySelector('body');
    // Calcul compliqué pour dire que le canvas doit prendre le maximum de place sur l'écran
    const scale = Math.min((body.clientHeight / app.canvas.height), (body.clientWidth / app.canvas.width));
    app.canvas.style.transform = 'scale('+scale+')';
}

function clicCanvas(event){
    if(event.target !== app.canvas) return;                             //Si la cible du clic n'est pas le canvas, on ne s'occupe pas du clic
    event.preventDefault();                                             //On annule le comportement par défaut du clic
    let x = event.offsetX;                                              //Position Horizontale du clic sur le canvas
    let y = event.offsetY;                                              //Position Verticale du clic sur le canvas
    //Clic Replay
    if(clicReplay(x, y)){ newParty(); return; }                         //Si clic sur le bouton replay, on lance une nouvelle partie
    //Clic Carte
    let indexCard = clicCard(x, y);                                     //On vérifie s'il y a clic sur une carte
    if(indexCard !== false){ selectCard(indexCard); return; }           //Si oui, on appelle la fonction de selection de carte
}

function clicReplay(x,y) {
    if(app.score === 0 || app.cards.length === app.idCardsFound.length){
        if(x >= 124 && x <= 124 + 72 && y >= 235 && y <= 235 + 16){ return true; }
    } else {
        if(x >= 279 && x <= 279 + 26 && y >= 15 && y <= 15 + 26){ return true; }
    }    
    return false;
}

function clicCard(x, y) {
    const w = config.cardSize.width;
    const h = config.cardSize.height;
    for(let index in app.cards){
        let cardX = 18 + 36 * (index % 8);                              //Position Horizontale de la carte (position la plus à gauche du plateau + décallage de 36 * n° de la colonne de la carte)
        let cardY = 95 + 52 * Math.floor(index/8);                      //Position Verticale de la carte (position la plus en haut du plateau + décallage de 52 * n° de la ligne de la carte)
        if(x >= cardX && x <= cardX + w && y >= cardY && y <= cardY + h){ //Si la position du clic correspond à celle d'une carte
            return index;                                               //On arrête de chercher et on renvoi l'index de la carte
        } 
    }
    return false;                                                       //Si aucune carte n'est dans la zone de clic, on renvoit false;
}

function selectCard(indexCard){
    // On annule la fonction si plus de point pour jouer ou carte déjà face visible 
    if(app.score <= 0 || app.idCardsSelected.includes(indexCard) ||   
    app.idCardsFound.includes(indexCard)) return; 
    app.idCardsSelected.push(indexCard);    // Ajoute la carte à celles sélectionnées
    app.score--; 		// Décrémente le score pour chaque sélection    
    checkPair(); 		// Vérifie si une paire a été formée
    render(); 		// Met à jour l'affichage
}

function checkPair() {
    if(app.idCardsSelected.length > 2){     //S'il y a plus de 2 cartes de sélectionnées, c'est que les 2 premières cartes ne formaient pas une paire
        app.idCardsSelected.splice(0,2);    //On supprime alors les 2 premières valeurs du tableau des cartes sélectionnées
    } 
    if(app.idCardsSelected.length === 2){   //S'il y a exactement 2 cartes de sélectionnées, on vérifie si c'est une paire
        let idCard1 = app.cards[app.idCardsSelected[0]];    //Id de la première carte sélectionnée
        let idCard2 = app.cards[app.idCardsSelected[1]];    //Id de la deuxième carte sélectionnée
        if(idCard1 === idCard2){ //Si les 2 cartes ont le même id, c'est qu'il s'agit d'une paire
            app.idCardsFound.push(app.idCardsSelected[0], app.idCardsSelected[1]); //On ajoute alors les 2 cartes au tableau des cartes trouvées
            app.idCardsSelected = []; //Puis on vide le tableau des cartes sélectionnées
            app.score += 5; //On ajoute 5 points au score car on a trouvé une paire
        }
    }
}

// Chargement des images
function loadImages() {                                                 // Fonction qui chargera toutes les images nécessaires en même temps
    for(let imgName of config.imagesList){                              // Boucle FOR sur la liste des images pour récupérer leur nom
        const img = new Image();                                        // Création d'une variable qui va stocker l'image à télécharger
        img.onload = () => saveImage(imgName, img);                     // Si l'image est bien téléchargée, on lance la fonction saveImage
        img.src = './images/'+ imgName;                                 // Lien vers l'image à télécharger
    }
}

function saveImage(nameComplet, img) {                                  // Fonction pour sauvegarder les images dans app.images
    const name = nameComplet.split(".");                                // On découpe le nom de l'image sous forme de tableau ["memory", "png"]
    app.images[name[0]] = img;                                          // On stocke l'image dans app.images.memory
    if(allImagesLoaded()){ newParty(); }                                // Si toutes les images sont téléchargées, on peut lancer la boucle de jeu
}

function allImagesLoaded() {
    let length = 0;                                                     
    for(let i in app.images){ length++; }                               //On calcul le nombre d'images enregistrée dans app.images
    return config.imagesList.length === length;                         //Si il y a autant d'images à télécharger que d'image enregistré c'est que c'est fini
}

// Nouvelle party
function newParty() {
    app.score = config.scoreInitial;
    app.cards = [];                                                     //On crée un tableau vide pour stocker les cartes de la partie
    app.idCardsSelected = [];                                           //On crée un tableau qui contiendra l'ID des cartes selectionnées
    app.idCardsFound = [];                                              //On crée un tableau qui contiendra l'ID des paires trouvées
    while(app.cards.length < config.nbCards){                           //On boucle tant qu'il n'y a pas le nombre de cartes attendu dans le tableau (ici 32 cartes)
        let newIdCard = Math.floor(Math.random()*52);                   //On choisi une carte parmi les 52 disponibles
        if(app.cards.indexOf(newIdCard) < 0){                           //Si la carte n'a pas déjà été choisie précédemment
            app.cards.push(newIdCard, newIdCard);                       //On l'ajoute 2 fois aux carte de la partie pour faire une paire
        }
    }  
    app.cards = app.cards.sort(() => Math.random() - 0.5);              //On mélange le tableau pour que les paires ne se suivent pas          
    render();      
}

function render() {
    app.context.drawImage(app.images['memory'], 0, 0);                  //A chaque nouveau rendu, on dessine l'image de fond pour recouvrir le canvas (et ne pas avoir à l'effacer)
    drawCards();
    drawScore();
    drawPopup();
}

function drawCards(){
    const w = config.cardSize.width;
    const h = config.cardSize.height;
    for(let index in app.cards){                                        //On passe en revue chacune des cartes grâce à une boucle
        let x = 18 + 36 * (index % 8);                                  //Position Horizontale de la carte (position la plus à gauche du plateau + décallage de 36 * n° de la colonne de la carte)
        let y = 95 + 52 * Math.floor(index/8);                          //Position Verticale de la carte (position la plus en haut du plateau + décallage de 52 * n° de la ligne de la carte)
        let id = app.cards[index];                                      //Id de la carte
        let imgX = w * (id % 13);                                       //Position Horizontale de la carte sur cards.png
        let imgY = h * Math.floor(id / 13);                             //Position Verticale de la carte sur cards.png        
        //Si la carte est selectionnée ou quelle fait partie des paire trouvée, on dessine la face de la carte
        //Sinon on dessine le dos de carte
        if(app.idCardsSelected.includes(index) || app.idCardsFound.includes(index)){
            app.context.drawImage(app.images['cards'], imgX, imgY, w, h, x, y, w, h);
        }else{
            app.context.drawImage(app.images['cards'], 0, 192, w, h, x, y, w, h);
        }
    }
}

function drawScore(){
    let scoreText = app.score.toString();
    let scoreWidth = scoreText.length * 7 + 1;
    let x = 77 - scoreWidth;
    let y = 22;
    const img = app.images['text'];
    for(let index in scoreText){
        let char = scoreText[index];
        app.context.drawImage(img, parseInt(char*8), 5, 8, 12, x + (index * 7), y, 8, 12);
    }
}

function drawPopup(){
    let x = 89;
    let y = 146;
    const img = app.images["popup"];
    if(app.score === 0){ //Si le score n'a pas atteint zéro, le joueur a perdu
        app.context.drawImage(img, 0, 106, 142, 106, x, y, 142, 106);
    }else if(app.cards.length === app.idCardsFound.length){ //S'il y a autant de carte dans le jeu que de cartes trouvées, c'est que le joueur à gagné
        app.context.drawImage(img, 0, 0, 142, 106, x, y, 142, 106);
    }
}